#include <Wire.h>

#include <LiquidCrystal_I2C.h>



// ================================================================================
// TMACOINBOT v0.01
// This firmware controls a DF600G coin machine. The user can send a request from
// a host PC to the Arduino and the Arduino will send a response back to the host
// PC with the number of coins and total amount  inserted.
// Written by Rein Velt (rein@mechanicape.com)  @ 20140506
//
// HARDWARE
// --------
//   SEEEDUINO MEGA
//   COIN MACHINE  (DF600F)                   -> connected to Serial#3 RX
//   LCD 2x16 I2C  (DFRobot Lcd module v1.1)  -> connected to SDA, SCL, GND ,+5V
//   HOST COMPUTER (PC/RPI/...)               -> connected to USB
// ================================================================================

//global constants
#define firmwareVersion      1

#define CoinmachineBaudrate  9600
#define HostBaudrate         19200

#define coinBufferLength     99 //max number of coins per transaction

//the 'token' is the value we receive from the coin machine when inserting a valid coin
//e.g. when we insert a coin of 50 cents the coin machine returns '10' over the serial port
#define tokenCoin1           1  //5 cent
#define tokenCoin2           2  //10 cent
#define tokenCoin3           4  //20 cent
#define tokenCoin4           10 //50 cent
#define tokenCoin5           20 //1 euro
#define tokenCoin6           40 //2 euro

#define valueCoin1           5  //5 cent
#define valueCoin2           10  //10 cent
#define valueCoin3           20  //20 cent
#define valueCoin4           50 //50 cent
#define valueCoin5           100 //1 euro
#define valueCoin6           200 //2 euro


//magic numbers are bad, so we replace them by magic constants ;-)
#define charLF               10 // ascii linefeed
#define charC                67 // ascii C
#define charR                82 // ascii R
#define charT                84 // ascii T

#define lcdAddress           0x20 //i2c address of the lcd
#define lcdRows              2  //number of rows on the lcd
#define lcdColumns           16 //number of columns on the lcd

//global variables
int currentCoinArr[coinBufferLength];  //array holding coins for the current transaction
int currentCoinPtr=0;
LiquidCrystal_I2C lcd(lcdAddress,lcdColumns,lcdRows);





//program initialisation
void setup()
{
  Serial3.begin(CoinmachineBaudrate); //init coinmachine baudrate
  Serial.begin(HostBaudrate); //init console baudrate
  
  Serial.print("#APECOINBOT COIN MODULE V");
  Serial.println(firmwareVersion);
  Serial.println("#  Usage: ");
  Serial.println("#\tCC - returns individual coins in current transaction");
  Serial.println("#\tCT - return the total amount of money in the current transaction");
  Serial.println("#\tCR - reset the current transaction");
  Serial.println("#  all commands must be terminated with a linefeed (ascii 10) ");
  Serial.println("#  The firmware will send a response followed by 'OK' + linefeed");
  lcd.init();
  lcd.backlight();
  lcd.home();
  lcd.print("** TMACOINBOT **"); 
  lcd.setCursor(0,1);
  lcd.print("insert coin_");
  
  
  
  
}


//main loop
void loop()
{
  //handle incoming request from host computer
  handleHostRequest();
  
   //handle coins from coinmachine
  handleCoinRequest();
 
}








//helper functions


void handleCoinRequest()
{
  //read data from the coin machine
  if (Serial3.available())
  {
    
    int inbyte=Serial3.read();
    currentCoinArr[currentCoinPtr]=inbyte;
    currentCoinPtr++;
    //update lcd
    int total=0; //cents
    for (int i=0;i<currentCoinPtr;i++)
    {
      total=total+mapTokenToCoin(currentCoinArr[i]);
    }
    lcd.clear();
    lcd.setCursor(0,0);
    lcd.print("COINS: ");
    lcd.print(currentCoinPtr);
    lcd.setCursor(0,1);
    lcd.print("TOTAL: ");
    lcd.print(total);
    lcd.print(" ");
    lcd.print("cent");
    
  }
}

void handleHostRequest()
{
  //read data from the host
  if (Serial.available()>2)
  {
      int requestCommand=Serial.read();    //the command
      int requestValue=Serial.read();      //a value byte or a dummy byte
      int requestTerminator=Serial.read(); //char(10) to terminate the request
      if (requestTerminator==charLF)
      {
          if (requestCommand==charC)  //C
          {
             switch (requestValue)
             {
               case charC: //CC - current coins
                 handleRequestCC();
                 break;
               case charT: //CT - current total
                 handleRequestCT();
                 break;
               case charR: //CR - current reset
                 handleRequestCR();
                 break;
             }  
          }
          Serial.println("OK");
      }
  }
}

void handleRequestCC()
{
  //create host response with individual coins in current transaction
  if (currentCoinPtr==0)
  {
    Serial.println("0"); //no coins
  }
  else
  {
    for (int i=0;i<currentCoinPtr;i++)
    {
      Serial.print(mapTokenToCoin(currentCoinArr[i]));
      Serial.print(" "); //separator between values
    }
    Serial.println();
  }
}

void handleRequestCT()
{
  //create host response with sum (in cents) in current transaction
   int total=0; //cents
   for (int i=0;i<currentCoinPtr;i++)
   {
     total=total+mapTokenToCoin(currentCoinArr[i]);
   }
   Serial.print(total);
   Serial.println(" cent");
   
}

void handleRequestCR()
{
  //reset the current transaction counter
  currentCoinPtr=0;
  lcd.clear();
  lcd.print("** TMACOINBOT **"); 
  lcd.setCursor(0,1);
  lcd.print("insert coin_");
}

int mapTokenToCoin(int token)
{
   int coin=0; //in cents
   switch(token)
   { 
     case tokenCoin1: coin=valueCoin1; break;
     case tokenCoin2: coin=valueCoin2; break;
     case tokenCoin3: coin=valueCoin3; break;
     case tokenCoin4: coin=valueCoin4; break;
     case tokenCoin5: coin=valueCoin5; break;
     case tokenCoin6: coin=valueCoin6; break;
   } 
   return coin;
}

